<?php
/**
 * Plugin Name: WP Monitor Client
 * Description: 管理用サイトにWordPress更新状況を毎日送信するクライアントプラグイン。
 * Version:     1.2.2
 * Author:      right hand合同会社
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/*
 * 管理用WordPress側のエンドポイントURLに書き換えてください。
 * 例: https://monitor.rh-ms.com/wp-json/wpm-monitor/v1/report
 */
if ( ! defined( 'WPM_MONITOR_ENDPOINT_URL' ) ) {
	define( 'WPM_MONITOR_ENDPOINT_URL', 'https://monitor.rh-ms.com/wp-json/wpm-monitor/v1/report' );
}

/*
 * 管理用プラグインと同じAPIキーを設定してください。
 */
if ( ! defined( 'WPM_MONITOR_API_KEY' ) ) {
	define( 'WPM_MONITOR_API_KEY', '' );
}

/*
 * 非公開配布用の更新メタデータJSONのURL。
 * 例: https://monitor.rh-ms.com/wpm-monitor-client.json
 */
if ( ! defined( 'WPM_MONITOR_UPDATE_JSON_URL' ) ) {
	define( 'WPM_MONITOR_UPDATE_JSON_URL', 'https://monitor.rh-ms.com/wpm-monitor-client.json' );
}

/*
 * 必要に応じてサイトIDを手動設定できます（空の場合はドメイン名を使用）。
 * 例: define( 'WPM_MONITOR_SITE_ID', 'rh-ms.com' );
 */
if ( ! defined( 'WPM_MONITOR_SITE_ID' ) ) {
	define( 'WPM_MONITOR_SITE_ID', '' );
}

/*
 * クライアント識別情報（未定義ならデフォルト値をセット）。
 */
if ( ! defined( 'WPM_MONITOR_CLIENT_KEY' ) ) {
	// 内部識別用の短いキー（英数字とハイフン程度）
	define( 'WPM_MONITOR_CLIENT_KEY', 'rh-ms' );
}

if ( ! defined( 'WPM_MONITOR_CLIENT_NAME' ) ) {
	// 表示用のクライアント名
	define( 'WPM_MONITOR_CLIENT_NAME', 'right hand合同会社' );
}

if ( ! defined( 'WPM_MONITOR_SITE_LABEL' ) ) {
	// クライアント内でのサイト名
	define( 'WPM_MONITOR_SITE_LABEL', '会社サイト' );
}

/**
 * サイトID・ドメイン・パスを取得。
 *
 * @return array{string,string,string} [ $site_id, $domain, $path ]
 */
function wpm_monitor_client_get_site_and_domain() {
	$home_url = home_url();
	$parsed   = wp_parse_url( $home_url );
	if ( ! is_array( $parsed ) ) {
		$parsed = array();
	}

	$domain = isset( $parsed['host'] ) ? $parsed['host'] : $home_url;
	if ( ! is_string( $domain ) || $domain === '' ) {
		$domain = $home_url;
	}
	$path   = isset( $parsed['path'] ) ? $parsed['path'] : '/';
	if ( ! is_string( $path ) || $path === '' ) {
		$path = '/';
	}
	$path = sanitize_text_field( $path );

	$default_site_id = trailingslashit( $home_url );

	$option = get_option( 'wpm_monitor_site_id', '' );
	if ( ! is_string( $option ) ) {
		$option = '';
	}
	$option = trim( $option );

	$site_id = $option !== '' ? $option : ( WPM_MONITOR_SITE_ID !== '' ? WPM_MONITOR_SITE_ID : $default_site_id );
	$site_id = sanitize_text_field( $site_id );
	/**
	 * @param string $site_id 送信するサイトID。
	 * @param string $domain  デフォルトのドメイン。
	 */
	$site_id = apply_filters( 'wpm_monitor_site_id', $site_id, $domain );

	return array( $site_id, $domain, $path );
}

/**
 * クライアント関連の設定値を取得（設定値 > 定数）。
 *
 * @return array{string,string,string} [ $client_key, $client_name, $site_label ]
 */
function wpm_monitor_client_get_client_meta() {
	// client_key
	$client_key = get_option( 'wpm_monitor_client_key', '' );
	if ( ! is_string( $client_key ) ) {
		$client_key = '';
	}
	$client_key = wpm_monitor_client_sanitize_client_key( $client_key );
	if ( $client_key === '' ) {
		$client_key = wpm_monitor_client_sanitize_client_key( WPM_MONITOR_CLIENT_KEY );
	}

	// client_name
	$client_name = get_option( 'wpm_monitor_client_name', '' );
	if ( ! is_string( $client_name ) ) {
		$client_name = '';
	}
	$client_name = wpm_monitor_client_sanitize_label( $client_name );
	if ( $client_name === '' ) {
		$client_name = wpm_monitor_client_sanitize_label( WPM_MONITOR_CLIENT_NAME );
	}

	// site_label
	$site_label = get_option( 'wpm_monitor_site_label', '' );
	if ( ! is_string( $site_label ) ) {
		$site_label = '';
	}
	$site_label = wpm_monitor_client_sanitize_label( $site_label );
	if ( $site_label === '' ) {
		$site_label = wpm_monitor_client_sanitize_label( WPM_MONITOR_SITE_LABEL );
	}

	return array( $client_key, $client_name, $site_label );
}

/**
 * APIキーを取得（設定値 > 定数）。
 *
 * @return string
 */
function wpm_monitor_client_get_api_key() {
	$api_key = get_option( 'wpm_monitor_api_key', '' );
	if ( ! is_string( $api_key ) ) {
		$api_key = '';
	}
	$api_key = wpm_monitor_client_sanitize_label( $api_key );
	if ( $api_key === '' ) {
		$api_key = wpm_monitor_client_sanitize_label( WPM_MONITOR_API_KEY );
	}

	return $api_key;
}

/**
 * 手動実行用: https://your-site/wp-cron.php?wpm_monitor_run=1&key=APIKEY
 * WP-CLIが使えない環境でも即座に送信できるようにする。
 */
function wpm_monitor_client_manual_run() {
	if ( ! isset( $_GET['wpm_monitor_run'], $_GET['key'] ) ) {
		return;
	}

	$key = sanitize_text_field( wp_unslash( $_GET['key'] ) );
	$api_key = wpm_monitor_client_get_api_key();

	if ( ! is_string( $key ) || $api_key === '' || ! hash_equals( $api_key, $key ) ) {
		status_header( 403 );
		exit( 'Forbidden' );
	}

	wpm_monitor_client_send_report();

	status_header( 200 );
	exit( 'ok' );
}
add_action( 'init', 'wpm_monitor_client_manual_run' );

/**
 * サイトID設定のサニタイズ。
 */
function wpm_monitor_client_sanitize_site_id( $value ) {
	if ( ! is_string( $value ) ) {
		return '';
	}
	$value = sanitize_text_field( trim( $value ) );
	if ( strlen( $value ) > 190 ) {
		$value = substr( $value, 0, 190 );
	}
	return $value;
}

/**
 * クライアントキーのサニタイズ。
 */
function wpm_monitor_client_sanitize_client_key( $value ) {
	if ( ! is_string( $value ) ) {
		return '';
	}
	$value = sanitize_key( $value );
	if ( strlen( $value ) > 100 ) {
		$value = substr( $value, 0, 100 );
	}
	return $value;
}

/**
 * クライアント名/サイト名のサニタイズ。
 */
function wpm_monitor_client_sanitize_label( $value ) {
	if ( ! is_string( $value ) ) {
		return '';
	}
	$value = sanitize_text_field( trim( $value ) );
	if ( strlen( $value ) > 191 ) {
		$value = substr( $value, 0, 191 );
	}
	return $value;
}

/**
 * 設定フィールドの描画。
 */
function wpm_monitor_client_render_site_id_field() {
	$current = get_option( 'wpm_monitor_site_id', '' );
	if ( ! is_string( $current ) ) {
		$current = '';
	}
	$current = sanitize_text_field( $current );

	$home_url = home_url();
	$placeholder = trailingslashit( $home_url );
	?>
	<input
		type="text"
		name="wpm_monitor_site_id"
		id="wpm_monitor_site_id"
		value="<?php echo esc_attr( $current ); ?>"
		class="regular-text"
		placeholder="<?php echo esc_attr( $placeholder ); ?>"
	/>
	<p class="description">
		空欄の場合はサイトURL（home_urlの末尾スラッシュ付き）を送信します。
	</p>
	<?php
}

/**
 * クライアントキーの描画。
 */
function wpm_monitor_client_render_client_key_field() {
	$current = get_option( 'wpm_monitor_client_key', '' );
	if ( ! is_string( $current ) ) {
		$current = '';
	}
	$current     = wpm_monitor_client_sanitize_client_key( $current );
	$placeholder = WPM_MONITOR_CLIENT_KEY;
	?>
	<input
		type="text"
		name="wpm_monitor_client_key"
		id="wpm_monitor_client_key"
		value="<?php echo esc_attr( $current ); ?>"
		class="regular-text"
		placeholder="<?php echo esc_attr( $placeholder ); ?>"
	/>
	<p class="description">
		未入力の場合は定数 WPM_MONITOR_CLIENT_KEY の値を使用します。
	</p>
	<?php
}

/**
 * クライアント名の描画。
 */
function wpm_monitor_client_render_client_name_field() {
	$current = get_option( 'wpm_monitor_client_name', '' );
	if ( ! is_string( $current ) ) {
		$current = '';
	}
	$current     = wpm_monitor_client_sanitize_label( $current );
	$placeholder = WPM_MONITOR_CLIENT_NAME;
	?>
	<input
		type="text"
		name="wpm_monitor_client_name"
		id="wpm_monitor_client_name"
		value="<?php echo esc_attr( $current ); ?>"
		class="regular-text"
		placeholder="<?php echo esc_attr( $placeholder ); ?>"
	/>
	<p class="description">
		未入力の場合は定数 WPM_MONITOR_CLIENT_NAME の値を使用します。
	</p>
	<?php
}

/**
 * サイト表示名の描画。
 */
function wpm_monitor_client_render_site_label_field() {
	$current = get_option( 'wpm_monitor_site_label', '' );
	if ( ! is_string( $current ) ) {
		$current = '';
	}
	$current     = wpm_monitor_client_sanitize_label( $current );
	$placeholder = WPM_MONITOR_SITE_LABEL;
	?>
	<input
		type="text"
		name="wpm_monitor_site_label"
		id="wpm_monitor_site_label"
		value="<?php echo esc_attr( $current ); ?>"
		class="regular-text"
		placeholder="<?php echo esc_attr( $placeholder ); ?>"
	/>
	<p class="description">
		未入力の場合は定数 WPM_MONITOR_SITE_LABEL の値を使用します。
	</p>
	<?php
}

/**
 * APIキーの描画。
 */
function wpm_monitor_client_render_api_key_field() {
	$current = get_option( 'wpm_monitor_api_key', '' );
	if ( ! is_string( $current ) ) {
		$current = '';
	}
	$current     = wpm_monitor_client_sanitize_label( $current );
	$placeholder = WPM_MONITOR_API_KEY;
	?>
	<input
		type="text"
		name="wpm_monitor_api_key"
		id="wpm_monitor_api_key"
		value="<?php echo esc_attr( $current ); ?>"
		class="regular-text"
		placeholder="<?php echo esc_attr( $placeholder ); ?>"
	/>
	<p class="description">
		未入力の場合は定数 WPM_MONITOR_API_KEY の値を使用します。
	</p>
	<?php
}

/**
 * 設定登録。
 */
function wpm_monitor_client_register_settings() {
	register_setting(
		'wpm_monitor_client',
		'wpm_monitor_site_id',
		array(
			'type'              => 'string',
			'sanitize_callback' => 'wpm_monitor_client_sanitize_site_id',
			'default'           => '',
		)
	);

	register_setting(
		'wpm_monitor_client',
		'wpm_monitor_api_key',
		array(
			'type'              => 'string',
			'sanitize_callback' => 'wpm_monitor_client_sanitize_label',
			'default'           => '',
		)
	);

	register_setting(
		'wpm_monitor_client',
		'wpm_monitor_client_key',
		array(
			'type'              => 'string',
			'sanitize_callback' => 'wpm_monitor_client_sanitize_client_key',
			'default'           => '',
		)
	);

	register_setting(
		'wpm_monitor_client',
		'wpm_monitor_client_name',
		array(
			'type'              => 'string',
			'sanitize_callback' => 'wpm_monitor_client_sanitize_label',
			'default'           => '',
		)
	);

	register_setting(
		'wpm_monitor_client',
		'wpm_monitor_site_label',
		array(
			'type'              => 'string',
			'sanitize_callback' => 'wpm_monitor_client_sanitize_label',
			'default'           => '',
		)
	);

	add_settings_section(
		'wpm_monitor_client_main',
		'基本設定',
		'__return_false',
		'wpm-monitor-client'
	);

	add_settings_field(
		'wpm_monitor_site_id',
		'サイトID',
		'wpm_monitor_client_render_site_id_field',
		'wpm-monitor-client',
		'wpm_monitor_client_main'
	);

	add_settings_field(
		'wpm_monitor_api_key',
		'APIキー',
		'wpm_monitor_client_render_api_key_field',
		'wpm-monitor-client',
		'wpm_monitor_client_main'
	);

	add_settings_field(
		'wpm_monitor_client_key',
		'クライアントキー',
		'wpm_monitor_client_render_client_key_field',
		'wpm-monitor-client',
		'wpm_monitor_client_main'
	);

	add_settings_field(
		'wpm_monitor_client_name',
		'クライアント名',
		'wpm_monitor_client_render_client_name_field',
		'wpm-monitor-client',
		'wpm_monitor_client_main'
	);

	add_settings_field(
		'wpm_monitor_site_label',
		'サイト表示名',
		'wpm_monitor_client_render_site_label_field',
		'wpm-monitor-client',
		'wpm_monitor_client_main'
	);
}
add_action( 'admin_init', 'wpm_monitor_client_register_settings' );

/**
 * 設定画面追加。
 */
function wpm_monitor_client_add_settings_page() {
	add_options_page(
		'WP Monitor Client',
		'WP Monitor Client',
		'manage_options',
		'wpm-monitor-client',
		'wpm_monitor_client_render_settings_page'
	);
}
add_action( 'admin_menu', 'wpm_monitor_client_add_settings_page' );

/**
 * 設定画面の描画。
 */
function wpm_monitor_client_render_settings_page() {
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}
	?>
	<div class="wrap">
		<h1>WP Monitor Client</h1>
		<form method="post" action="options.php">
			<?php
				settings_fields( 'wpm_monitor_client' );
				do_settings_sections( 'wpm-monitor-client' );
				submit_button();
			?>
		</form>
	</div>
	<?php
}

/**
 * プラグイン一覧に設定リンクを表示。
 */
function wpm_monitor_client_action_links( $links ) {
	$url      = admin_url( 'options-general.php?page=wpm-monitor-client' );
	$settings = '<a href="' . esc_url( $url ) . '">' . esc_html__( 'Settings' ) . '</a>';
	array_unshift( $links, $settings );
	return $links;
}
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'wpm_monitor_client_action_links' );

/**
 * 非公開更新用のメタデータ取得。
 */
function wpm_monitor_client_get_update_info() {
	$cached = get_transient( 'wpm_monitor_client_update_info' );
	if ( $cached !== false ) {
		return $cached;
	}

	$url = WPM_MONITOR_UPDATE_JSON_URL;
	/**
	 * @param string $url 更新情報JSONのURL。
	 */
	$url = apply_filters( 'wpm_monitor_update_json_url', $url );

	$args = array(
		'timeout' => 10,
		'headers' => array(
			'Accept'        => 'application/json',
			'X-WPM-API-Key' => wpm_monitor_client_get_api_key(),
		),
	);

	$response = wp_remote_get( $url, $args );

	if ( is_wp_error( $response ) ) {
		error_log( '[wpm-monitor-client] update check error: ' . $response->get_error_message() );
		set_transient( 'wpm_monitor_client_update_info', array(), 15 * MINUTE_IN_SECONDS );
		return array();
	}

	$code = wp_remote_retrieve_response_code( $response );
	if ( $code < 200 || $code >= 300 ) {
		error_log( '[wpm-monitor-client] update check HTTP ' . $code );
		set_transient( 'wpm_monitor_client_update_info', array(), 15 * MINUTE_IN_SECONDS );
		return array();
	}

	$body = wp_remote_retrieve_body( $response );
	$data = json_decode( $body, true );

	if ( ! is_array( $data ) || empty( $data['version'] ) || empty( $data['download_url'] ) ) {
		error_log( '[wpm-monitor-client] update check invalid payload' );
		set_transient( 'wpm_monitor_client_update_info', array(), 15 * MINUTE_IN_SECONDS );
		return array();
	}

	set_transient( 'wpm_monitor_client_update_info', $data, 6 * HOUR_IN_SECONDS );
	return $data;
}

/**
 * 更新チェックに差し込む。
 */
function wpm_monitor_client_update_plugins_filter( $transient ) {
	if ( empty( $transient->checked ) ) {
		return $transient;
	}

	$plugin_file = plugin_basename( __FILE__ );
	$current     = isset( $transient->checked[ $plugin_file ] ) ? $transient->checked[ $plugin_file ] : null;
	if ( ! $current ) {
		return $transient;
	}

	$info = wpm_monitor_client_get_update_info();
	if ( empty( $info['version'] ) || empty( $info['download_url'] ) ) {
		return $transient;
	}

	if ( version_compare( $info['version'], $current, '<=' ) ) {
		return $transient;
	}

	$item = (object) array(
		'slug'        => 'wpm-monitor-client',
		'plugin'      => $plugin_file,
		'new_version' => $info['version'],
		'package'     => $info['download_url'],
		'url'         => ! empty( $info['homepage'] ) ? $info['homepage'] : 'https://monitor.rh-ms.com/',
		'tested'      => ! empty( $info['tested'] ) ? $info['tested'] : '',
		'requires'    => ! empty( $info['requires'] ) ? $info['requires'] : '',
		'icons'       => array(),
	);

	$transient->response[ $plugin_file ] = $item;
	return $transient;
}
add_filter( 'site_transient_update_plugins', 'wpm_monitor_client_update_plugins_filter' );

/**
 * 詳細ポップアップ用。
 */
function wpm_monitor_client_plugins_api( $res, $action, $args ) {
	if ( $action !== 'plugin_information' ) {
		return $res;
	}

	if ( empty( $args->slug ) || $args->slug !== 'wpm-monitor-client' ) {
		return $res;
	}

	$info = wpm_monitor_client_get_update_info();
	if ( empty( $info ) ) {
		return $res;
	}

	$res = (object) array(
		'name'          => 'WP Monitor Client',
		'slug'          => 'wpm-monitor-client',
		'version'       => isset( $info['version'] ) ? $info['version'] : '',
		'download_link' => isset( $info['download_url'] ) ? $info['download_url'] : '',
		'sections'      => array(
			'description' => isset( $info['changelog'] ) ? wp_kses_post( $info['changelog'] ) : 'Private distribution',
			'changelog'   => isset( $info['changelog'] ) ? wp_kses_post( $info['changelog'] ) : '',
		),
		'tested'        => isset( $info['tested'] ) ? $info['tested'] : '',
		'requires'      => isset( $info['requires'] ) ? $info['requires'] : '',
	);

	return $res;
}
add_filter( 'plugins_api', 'wpm_monitor_client_plugins_api', 10, 3 );

/**
 * このプラグインだけ自動更新を常に有効化（UIで選べなくても更新される）。
 */
add_filter(
	'auto_update_plugin',
	function ( $update, $item ) {
		if ( isset( $item->slug ) && $item->slug === 'wpm-monitor-client' ) {
			return true;
		}
		return $update;
	},
	10,
	2
);

/**
 * プラグイン有効化時：毎日実行のWP Cronを登録。
 */
function wpm_monitor_client_activate() {
	if ( ! wp_next_scheduled( 'wpm_monitor_client_daily_event' ) ) {
		// 今から5分後に初回実行、その後は毎日
		wp_schedule_event( time() + 300, 'daily', 'wpm_monitor_client_daily_event' );
	}
}
register_activation_hook( __FILE__, 'wpm_monitor_client_activate' );

/**
 * プラグイン停止時：WP Cronを解除。
 */
function wpm_monitor_client_deactivate() {
	$timestamp = wp_next_scheduled( 'wpm_monitor_client_daily_event' );
	if ( $timestamp ) {
		wp_unschedule_event( $timestamp, 'wpm_monitor_client_daily_event' );
	}
}
register_deactivation_hook( __FILE__, 'wpm_monitor_client_deactivate' );

/**
 * Cronフックに処理を紐付け。
 */
add_action( 'wpm_monitor_client_daily_event', 'wpm_monitor_client_send_report' );

/**
 * 更新状況を集計して管理側に送信。
 */
function wpm_monitor_client_send_report() {
	if ( ! defined( 'WPM_MONITOR_ENDPOINT_URL' ) ) {
		return;
	}
	$api_key = wpm_monitor_client_get_api_key();
	if ( $api_key === '' ) {
		return;
	}

	// ログインURL（カスタムログインでもフックで差し替え可）
	$login_url = function_exists( 'wp_login_url' ) ? wp_login_url() : site_url( 'wp-login.php' );
	/**
	 * @param string $login_url 送信するログインURL。
	 */
	$login_url = apply_filters( 'wpm_monitor_client_login_url', $login_url );

	// 更新チェック用の関数を読み込む
	if ( ! function_exists( 'wp_version_check' ) ) {
		require_once ABSPATH . 'wp-admin/includes/update.php';
	}
	if ( ! function_exists( 'get_plugins' ) ) {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
	}
	if ( ! function_exists( 'wp_get_themes' ) ) {
		require_once ABSPATH . 'wp-admin/includes/theme.php';
	}

	// コア・プラグイン・テーマの更新情報を最新化
	// （WordPress本体と同じ関数を使います） [oai_citation:3‡GitHub](https://github.com/WordPress/wordpress-develop/blob/trunk/src/wp-admin/includes/class-wp-debug-data.php?utm_source=chatgpt.com)
	wp_version_check();
	wp_update_plugins();
	wp_update_themes();

	// プラグインの更新件数
	$plugin_updates_count = 0;
	$update_plugins       = get_site_transient( 'update_plugins' );
	if ( is_object( $update_plugins ) && ! empty( $update_plugins->response ) && is_array( $update_plugins->response ) ) {
		$plugin_updates_count = count( $update_plugins->response );
	}

	// テーマの更新件数
	$theme_updates_count = 0;
	$update_themes       = get_site_transient( 'update_themes' );
	if ( is_object( $update_themes ) && ! empty( $update_themes->response ) && is_array( $update_themes->response ) ) {
		$theme_updates_count = count( $update_themes->response );
	}

	// コアの更新件数（通常 0 か 1 とします） [oai_citation:4‡WordPress Developer Resources](https://developer.wordpress.org/reference/functions/get_core_updates/?utm_source=chatgpt.com)
	$core_updates_count = 0;
	$core_updates       = get_core_updates();
	if ( is_array( $core_updates ) ) {
		foreach ( $core_updates as $update ) {
			if ( isset( $update->response ) && 'upgrade' === $update->response ) {
				$core_updates_count = 1;
				break;
			}
		}
	}

	// サイトIDとドメインを取得（設定値 > 定数 > ドメイン）
	list( $site_id, $domain, $path ) = wpm_monitor_client_get_site_and_domain();
	list( $client_key, $client_name, $site_label ) = wpm_monitor_client_get_client_meta();

	$payload = array(
		// 必要なら手動で識別子を変えてもOKです（例: client-001 など）
		'site_id'             => $site_id,
		'domain'              => $domain,
		'path'                => $path,
		'client_key'          => $client_key,
		'client_name'         => $client_name,
		'site_label'          => $site_label,
		'core_updates_count'  => $core_updates_count,
		'plugin_updates_count'=> $plugin_updates_count,
		'theme_updates_count' => $theme_updates_count,
		// ISO8601形式 (例: 2025-11-29T03:00:00Z)
		'checked_at'          => gmdate( 'c' ),
		'login_url'           => $login_url,
	);

	$args = array(
		'timeout' => 15,
		'headers' => array(
			'Content-Type'   => 'application/json',
			'X-WPM-API-Key'  => $api_key,
		),
		'body'    => wp_json_encode( $payload ),
	);

	$response = wp_remote_post( WPM_MONITOR_ENDPOINT_URL, $args );

	if ( is_wp_error( $response ) ) {
		// 必要ならログに残す
		error_log( '[wpm-monitor-client] HTTP error: ' . $response->get_error_message() );
		return;
	}

	$code = wp_remote_retrieve_response_code( $response );
	if ( $code < 200 || $code >= 300 ) {
		error_log( '[wpm-monitor-client] Unexpected response code: ' . $code );
	}
}
